# --- Nepieciešamās bibliotēkas ---
# Pārliecinieties, ka tās ir instalētas, terminālī izpildot:
# pip install pandas requests lxml shapely numpy
import pandas as pd
import numpy as np
from lxml import etree
from shapely.geometry import Polygon
from pathlib import Path
import sys
import re
import requests
import zipfile
import io

# --- === Vispārējā konfigurācija === ---

# 1. daļa: GML apstrāde un GPS ģenerēšana
# <<< LABOJUMS ŠEIT: Atjaunināta saite uz GML faila arhīvu >>>
building_zip_url = 'https://grafws.kadastrs.lv/atom/bu/20251117/Building.gml.zip'
input_gml_file = Path('Building.gml') # Faila nosaukums arhīvā ir ar lielo "B"

# 2. daļa: Ēku izslēgšana
exclusion_file_url = 'https://data.gov.lv/dati/dataset/09f883c4-8474-45d9-9b50-7d1b0ae5723a/resource/9461cdf9-cac9-458e-86f1-7c1f7721eb67/download/buves_daba_neeksiste_2025-08-04.csv'
local_exclusion_csv = Path('buves_daba_neeksiste_lejupieladets.csv')
input_summ_level_csv = Path('out-summ-level.csv')

# 3. daļa: Gala faili un atskaites
output_final_csv = Path('out-all.csv')
report_deleted_buildings = Path('atskaite-non-building.txt')
report_missing_gps = Path('atskaite-all.txt')

# --- Kolonnu nosaukumi ---
exclusion_csv_column_name = 'BuiCadNr'
csv_kadastrs_col = 'kadastrs'
csv_cilveki_col = 'kopejie_cilveki'
csv_level_col = 'level'
gps_kadastrs_col = 'ekas_kadastrs'
gps_point_col = 'building_point'
output_eka_col = 'eka'
output_point_col = 'point'
output_cilveki_col = 'cilveki'
output_level_col = 'level'

# XML Nosaukumvietas (Namespaces) GML failam
ns = {
    'bu': "http://inspire.ec.europa.eu/schemas/bu-core2d/4.0",
    'gml': "http://www.opengis.net/gml/3.2",
}

# --- === Skripta izpilde === ---

# --- 0. DAĻA: Sākotnējo datu sagatavošana (lejupielāde un atarhivēšana) ---
print("--- 0. DAĻA: Sākotnējo datu sagatavošana ---")
try:
    # Pārbaudām, vai Building.gml jau eksistē
    if input_gml_file.is_file():
        print(f"Info: Fails '{input_gml_file}' jau eksistē. Lejupielāde tiek izlaista.")
    else:
        print(f"Mēģinām lejupielādēt un atarhivēt '{input_gml_file}'...")
        print(f"Lejupielādējam no: {building_zip_url}")
        
        response = requests.get(building_zip_url, stream=True)
        response.raise_for_status()

        # Atarhivējam failu no arhīva atmiņā
        with zipfile.ZipFile(io.BytesIO(response.content)) as z:
            # Faila nosaukums arhīvā ir ar lielo burtu 'Building.gml'
            if 'Building.gml' in z.namelist():
                z.extract('Building.gml')
                print(f"Fails '{input_gml_file}' veiksmīgi lejupielādēts un atarhivēts.")
            else:
                raise FileNotFoundError(f"Arhīvā netika atrasts '{input_gml_file.name}' fails.")

    # Lejupielādējam izslēdzamo ēku sarakstu
    print(f"\nLejupielādējam izslēdzamo ēku sarakstu...")
    response_exclusion = requests.get(exclusion_file_url)
    response_exclusion.raise_for_status()
    with open(local_exclusion_csv, 'wb') as f: f.write(response_exclusion.content)
    print(f"Izslēdzamo ēku saraksts saglabāts kā: {local_exclusion_csv}")

except requests.exceptions.RequestException as e:
    print(f"KĻŪDA: Neizdevās lejupielādēt failu. Pārbaudiet interneta savienojumu vai URL adresi.", file=sys.stderr)
    print(f"Detalizētāk: {e}", file=sys.stderr)
    sys.exit(1)
except (zipfile.BadZipFile, FileNotFoundError) as e:
    print(f"KĻŪDA: Neizdevās atarhivēt failu.", file=sys.stderr)
    print(f"Detalizētāk: {e}", file=sys.stderr)
    sys.exit(1)
except Exception as e:
    print(f"KĻŪDA datu sagatavošanas posmā: {e}", file=sys.stderr)
    sys.exit(1)

# --- 1. DAĻA: GPS datu ģenerēšana no GML faila (atmiņā) ---
print("\n--- 1. DAĻA: GPS datu ģenerēšana no GML faila ---")
df_gps = pd.DataFrame()
try:
    print(f"Nolasām un parsējam GML failu: {input_gml_file}...")
    context = etree.iterparse(str(input_gml_file), events=('end',), tag='{'+ns['bu']+'}Building')
    
    results_gps = []
    processed_count = 0
    error_count = 0

    for event, elem in context:
        try:
            building_id_attr = elem.get('{'+ns['gml']+'}id')
            if not building_id_attr: continue

            match = re.search(r'(\d{14})$', building_id_attr)
            if not match: continue
            
            ekas_kadastrs_14 = match.group(1)

            pos_list_elem = elem.xpath('.//gml:posList', namespaces=ns)
            if not (pos_list_elem and pos_list_elem[0].text): continue

            numbers = [float(n) for n in pos_list_elem[0].text.strip().split()]
            if len(numbers) < 6 or len(numbers) % 2 != 0: continue

            points_for_shapely = list(zip(numbers[1::2], numbers[0::2])) # Lon, Lat
            polygon = Polygon(points_for_shapely).buffer(0)

            if polygon and polygon.geom_type == 'Polygon' and polygon.area > 1e-12:
                centroid_shapely = polygon.centroid
                building_point_str = f"POINT({centroid_shapely.y:.8f} {centroid_shapely.x:.8f})"
                results_gps.append({gps_kadastrs_col: ekas_kadastrs_14, gps_point_col: building_point_str})
                processed_count += 1
            else: error_count += 1
        except Exception: error_count += 1
        finally:
            elem.clear()
            while elem.getprevious() is not None: del elem.getparent()[0]

    if results_gps:
        df_gps = pd.DataFrame(results_gps)
        df_gps = df_gps.drop_duplicates(subset=[gps_kadastrs_col], keep='first')
        print(f"Veiksmīgi apstrādātas un atmiņā sagatavotas {len(df_gps)} ēkas ar GPS centroīdiem.")
        if error_count > 0: print(f"Ēkas ar kļūdām vai izlaistas GML apstrādē: {error_count}")
    else:
        print("KĻŪDA: No GML faila neizdevās nolasīt nevienu derīgu ēkas GPS punktu. Pārtraucam darbību.", file=sys.stderr)
        sys.exit(1)

except FileNotFoundError:
    print(f"KĻŪDA: Ievades GML fails '{input_gml_file}' nav atrasts!", file=sys.stderr)
    sys.exit(1)
except Exception as e:
    print(f"KĻŪDA GML faila apstrādes laikā: {e}", file=sys.stderr)
    sys.exit(1)


# --- 2. DAĻA: Nevajadzīgo ēku filtrēšana ---
print("\n--- 2. DAĻA: Nevajadzīgo ēku filtrēšana ---")
try:
    df_exclusion = pd.read_csv(local_exclusion_csv, usecols=[exclusion_csv_column_name], dtype=str)
    exclusion_kadastrs_set = set(df_exclusion[exclusion_csv_column_name].dropna().astype(str))
    print(f"Nolasīti {len(exclusion_kadastrs_set)} unikāli izslēdzamie kadastra numuri.")
    
    print(f"\nNolasām galveno datu failu: {input_summ_level_csv}...")
    df_main = pd.read_csv(input_summ_level_csv, dtype={csv_kadastrs_col: str, csv_cilveki_col: 'Int64', csv_level_col: str})
    print(f"Nolasītas {len(df_main)} rindas.")

    print("Veicam rindu filtrēšanu...")
    rows_to_remove_mask = df_main[csv_kadastrs_col].isin(exclusion_kadastrs_set)
    df_removed = df_main[rows_to_remove_mask].copy()
    df_kept = df_main[~rows_to_remove_mask].copy()

    removed_count = len(df_removed)
    kept_count = len(df_kept)
    print(f"Atrastas {removed_count} rindas izdzēšanai.")
    print(f"{kept_count} rindas tiek nodotas tālākai apstrādei.")

    with open(report_deleted_buildings, 'w', encoding='utf-8') as f:
        f.write(f"Atskaites par rindām, kas izdzēstas no '{input_summ_level_csv.name}'\n" + "="*80 + "\n\n")
        f.write(f"Kopējais izdzēsto rindu skaits: {removed_count}\n\n")
        if removed_count > 0: f.write(df_removed.to_csv(index=False, lineterminator='\n'))
    print(f"Atskaite par izdzēstajām ēkām saglabāta: {report_deleted_buildings}")

except FileNotFoundError:
    print(f"KĻŪDA: Fails '{input_summ_level_csv}' vai '{local_exclusion_csv}' nav atrasts!", file=sys.stderr)
    sys.exit(1)
except Exception as e:
    print(f"Kļūda datu filtrēšanas posmā: {e}", file=sys.stderr)
    sys.exit(1)

# --- 3. DAĻA: Datu apvienošana un gala faila veidošana ---
print("\n--- 3. DAĻA: Datu apvienošana un gala faila veidošana ---")
print("Apvienojam datus...")
merged_df = pd.merge(df_kept, df_gps, how='left', left_on=csv_kadastrs_col, right_on=gps_kadastrs_col)
print(f"Datu apvienošana pabeigta. Rezultātā {len(merged_df)} rindas.")

df_output_temp = merged_df.rename(columns={
    csv_kadastrs_col: output_eka_col,
    csv_cilveki_col: output_cilveki_col,
    csv_level_col: output_level_col,
    gps_point_col: output_point_col
})
final_columns = [output_eka_col, output_point_col, output_cilveki_col, output_level_col]
df_output_temp = df_output_temp[final_columns]

print("Meklējam un dzēšam rindas bez GPS punkta...")
missing_point_mask = df_output_temp[output_point_col].isna()
df_missing_report = df_output_temp[missing_point_mask].copy()
df_output_final = df_output_temp[~missing_point_mask].copy()

missing_count = len(df_missing_report)
final_count = len(df_output_final)
print(f"Atrastas un izdzēstas {missing_count} rindas bez atbilstoša GPS punkta.")
print(f"Gala failā tiks saglabātas {final_count} rindas.")

try:
    df_output_final.to_csv(output_final_csv, index=False, sep=',', na_rep='')
    print(f"\nGala rezultāts veiksmīgi saglabāts failā: {output_final_csv}")
except Exception as e:
    print(f"KĻŪDA saglabājot gala CSV failu {output_final_csv}: {e}", file=sys.stderr)

try:
    with open(report_missing_gps, 'w', encoding='utf-8') as f:
        f.write(f"Atskaites par rindām, kurām netika atrasts GPS punkts\n" + "="*80 + "\n\n")
        f.write(f"Kopējais rindu skaits bez GPS punkta: {missing_count}\n\n")
        if missing_count > 0:
            for eka_kad in df_missing_report[output_eka_col]:
                f.write(f"{eka_kad}\n")
    print(f"Atskaite par rindām bez GPS saglabāta: {report_missing_gps}")
except Exception as e_report:
    print(f"KĻŪDA rakstot atskaites failu {report_missing_gps}: {e_report}", file=sys.stderr)

print("\nSkripta darbība pabeigta.")